# IMB Nekretnine - Real Estate Website

A bilingual (Croatian/English) real estate website for IMB Nekretnine, featuring property listings in the Dubrovnik region. Built with Laravel 12, this application integrates with a legacy property management system while providing a modern web interface.

![Laravel](https://img.shields.io/badge/Laravel-12.x-FF2D20?style=flat&logo=laravel)
![PHP](https://img.shields.io/badge/PHP-8.2+-777BB4?style=flat&logo=php)
![Tailwind CSS](https://img.shields.io/badge/Tailwind_CSS-4.0-38B2AC?style=flat&logo=tailwind-css)
![License](https://img.shields.io/badge/License-MIT-green.svg)

## Table of Contents

- [Features](#features)
- [System Requirements](#system-requirements)
- [Local Development Setup](#local-development-setup)
- [Server Deployment](#server-deployment)
- [Project Architecture](#project-architecture)
- [Development Workflow](#development-workflow)
- [Testing](#testing)
- [Troubleshooting](#troubleshooting)
- [License](#license)

---

## Features

- **Bilingual Interface**: Full Croatian (primary) and English language support
- **Property Search**: Advanced filtering by location, type, price range, and features
- **Legacy Integration**: Seamless integration with existing `baza/` property management system
- **Responsive Design**: Mobile-first design using Tailwind CSS 4.0
- **Contact Forms**: Property-specific and general inquiry forms with email notifications
- **SEO-Friendly URLs**: Multilingual URL structure with clean aliases
- **Image Gallery**: Dynamic image serving with security protection against path traversal
- **Queue System**: Background job processing for emails and heavy operations

---

## System Requirements

### Minimum Requirements

- **PHP**: 8.2 or higher
- **Composer**: 2.x
- **Node.js**: 18.x or higher
- **NPM**: 9.x or higher
- **MySQL**: 8.0 or higher
- **Web Server**: Apache 2.4+ or Nginx 1.18+

### PHP Extensions Required

```
php-cli
php-curl
php-mbstring
php-xml
php-zip
php-mysql
php-gd
php-json
php-bcmath
php-tokenizer
php-fileinfo
```

### Optional Dependencies

- **Redis**: For cache and session storage (recommended for production)
- **Supervisor**: For queue worker management (recommended for production)

---

## Local Development Setup

### Step 1: Clone the Repository

```bash
git clone https://github.com/your-repo/imb-real-estate.git
cd imb-real-estate
```

### Step 2: Install Dependencies

The project includes a convenient setup script that handles all initial configuration:

```bash
composer setup
```

This command will:
1. Install PHP dependencies via Composer
2. Copy `.env.example` to `.env` (if not exists)
3. Generate application key
4. Run database migrations
5. Install Node.js dependencies
6. Build frontend assets

### Step 3: Configure Environment

Edit the `.env` file with your local database credentials:

```env
# Application
APP_NAME="IMB Nekretnine"
APP_ENV=local
APP_DEBUG=true
APP_URL=http://localhost:8000

# Database
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=imb
DB_USERNAME=root
DB_PASSWORD=your_password

# Cache & Session
CACHE_STORE=database
SESSION_DRIVER=database
QUEUE_CONNECTION=database

# Mail (Development)
MAIL_MAILER=log
```

### Step 4: Create Database

Create the MySQL database:

```bash
mysql -u root -p
```

```sql
CREATE DATABASE imb CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
EXIT;
```

### Step 5: Set Up Legacy "baza" Directory

The application requires the legacy `baza/` directory containing:
- Property management system files
- Property images in `baza/files/realestate/{company_id}/{realestate_id}/web_images/`
- Legacy database tables (prefixed with `baza_`)

**Important**: The `baza/` directory is git-ignored. Obtain it from the existing system or backup.

### Step 6: Import Legacy Database Tables

Import the required `baza_*` tables into your database:

```bash
mysql -u root -p imb < database/legacy_schema.sql
```

Required tables:
- `baza_realestate` - Core property data
- `baza_realestate_web` - Web-specific data
- `baza_locations_all` - Detailed locations
- `baza_locations_main` - Main location categories
- `baza_realestate_types` - Property types
- `baza_company` - Company information

### Step 7: Start Development Server

Start all development services with a single command:

```bash
composer dev
```

This starts:
- Laravel development server (http://localhost:8000)
- Queue worker for background jobs
- Laravel Pail for real-time logs
- Vite dev server with hot reload

**Alternative**: Start services individually:

```bash
# Terminal 1: Laravel server
php artisan serve

# Terminal 2: Queue worker
php artisan queue:listen --tries=1

# Terminal 3: Vite dev server
npm run dev
```

### Step 8: Access the Application

Open your browser and navigate to:
- **Homepage**: http://localhost:8000
- **Property Search**: http://localhost:8000/pretraga (Croatian) or /search (English)
- **Contact**: http://localhost:8000/kontakt (Croatian) or /contact (English)

---

## Server Deployment

### Prerequisites

- SSH access to server
- Root or sudo privileges
- Domain name pointing to server IP

### Step 1: Server Preparation

#### Update System

```bash
sudo apt update && sudo apt upgrade -y
```

#### Install Required Software

```bash
# PHP 8.2 and extensions
sudo apt install -y php8.2 php8.2-fpm php8.2-cli php8.2-mysql php8.2-xml \
    php8.2-mbstring php8.2-curl php8.2-zip php8.2-gd php8.2-bcmath php8.2-tokenizer

# MySQL
sudo apt install -y mysql-server

# Nginx
sudo apt install -y nginx

# Node.js (via NodeSource)
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt install -y nodejs

# Composer
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer

# Supervisor (for queue workers)
sudo apt install -y supervisor
```

### Step 2: Configure MySQL

```bash
sudo mysql_secure_installation
sudo mysql -u root -p
```

```sql
CREATE DATABASE imb_production CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'imb_user'@'localhost' IDENTIFIED BY 'secure_password_here';
GRANT ALL PRIVILEGES ON imb_production.* TO 'imb_user'@'localhost';
FLUSH PRIVILEGES;
EXIT;
```

### Step 3: Clone and Configure Application

```bash
# Create web directory
sudo mkdir -p /var/www/imbnekretnine
cd /var/www/imbnekretnine

# Clone repository
sudo git clone https://github.com/your-repo/imb-real-estate.git .

# Set permissions
sudo chown -R www-data:www-data /var/www/imbnekretnine
sudo chmod -R 755 /var/www/imbnekretnine/storage
sudo chmod -R 755 /var/www/imbnekretnine/bootstrap/cache
```

### Step 4: Configure Environment

```bash
# Copy and edit environment file
sudo cp .env.example .env
sudo nano .env
```

Production `.env` configuration:

```env
APP_NAME="IMB Nekretnine"
APP_ENV=production
APP_KEY=base64:... # Will be generated
APP_DEBUG=false
APP_URL=https://imbnekretnine.hr

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=imb_production
DB_USERNAME=imb_user
DB_PASSWORD=secure_password_here

CACHE_STORE=redis
SESSION_DRIVER=redis
QUEUE_CONNECTION=redis

REDIS_CLIENT=phpredis
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379

# Production Mail Configuration
MAIL_MAILER=smtp
MAIL_HOST=smtp.example.com
MAIL_PORT=587
MAIL_USERNAME=your_email@domain.com
MAIL_PASSWORD=your_email_password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS="info@imbnekretnine.hr"
MAIL_FROM_NAME="${APP_NAME}"
```

### Step 5: Install and Build

```bash
# Install dependencies (production only)
sudo -u www-data composer install --no-dev --optimize-autoloader

# Generate application key
sudo -u www-data php artisan key:generate

# Run migrations
sudo -u www-data php artisan migrate --force

# Build frontend assets
npm ci --production
npm run build

# Cache configuration
sudo -u www-data php artisan config:cache
sudo -u www-data php artisan route:cache
sudo -u www-data php artisan view:cache
```

### Step 6: Configure Nginx

Create Nginx configuration:

```bash
sudo nano /etc/nginx/sites-available/imbnekretnine
```

```nginx
server {
    listen 80;
    server_name imbnekretnine.hr www.imbnekretnine.hr;
    root /var/www/imbnekretnine/public;

    add_header X-Frame-Options "SAMEORIGIN";
    add_header X-Content-Type-Options "nosniff";

    index index.php;

    charset utf-8;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location = /favicon.ico { access_log off; log_not_found off; }
    location = /robots.txt  { access_log off; log_not_found off; }

    error_page 404 /index.php;

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }

    location ~ /\.(?!well-known).* {
        deny all;
    }

    # Serve baza files
    location /baza/ {
        root /var/www/imbnekretnine;
        internal;
    }
}
```

Enable site and restart Nginx:

```bash
sudo ln -s /etc/nginx/sites-available/imbnekretnine /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl restart nginx
sudo systemctl restart php8.2-fpm
```

### Step 7: Configure SSL with Let's Encrypt

```bash
sudo apt install -y certbot python3-certbot-nginx
sudo certbot --nginx -d imbnekretnine.hr -d www.imbnekretnine.hr
```

### Step 8: Configure Queue Workers with Supervisor

```bash
sudo nano /etc/supervisor/conf.d/imb-worker.conf
```

```ini
[program:imb-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /var/www/imbnekretnine/artisan queue:work --sleep=3 --tries=3 --max-time=3600
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/var/www/imbnekretnine/storage/logs/worker.log
stopwaitsecs=3600
```

Start supervisor:

```bash
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start imb-worker:*
```

### Step 9: Configure Cron for Scheduled Tasks

```bash
sudo crontab -e -u www-data
```

Add:

```cron
* * * * * cd /var/www/imbnekretnine && php artisan schedule:run >> /dev/null 2>&1
```

### Step 10: Set Up Deployment Script (Optional)

Create deployment script for easy updates:

```bash
sudo nano /var/www/imbnekretnine/deploy.sh
```

```bash
#!/bin/bash

cd /var/www/imbnekretnine

# Enable maintenance mode
php artisan down

# Pull latest changes
git pull origin main

# Install dependencies
composer install --no-dev --optimize-autoloader

# Run migrations
php artisan migrate --force

# Clear and cache
php artisan config:cache
php artisan route:cache
php artisan view:cache

# Restart queue workers
sudo supervisorctl restart imb-worker:*

# Disable maintenance mode
php artisan up

echo "Deployment completed successfully!"
```

Make executable:

```bash
sudo chmod +x /var/www/imbnekretnine/deploy.sh
```

---

## Project Architecture

### Directory Structure

```
imb-real-estate/
├── app/
│   ├── Console/          # Artisan commands
│   ├── Http/
│   │   ├── Controllers/  # Request handlers
│   │   └── Middleware/   # HTTP middleware
│   ├── Mail/             # Email classes
│   ├── Models/           # Eloquent models
│   └── Helpers.php       # Global helper functions
├── baza/                 # Legacy system (git-ignored)
│   └── files/
│       └── realestate/   # Property images
├── bootstrap/
├── config/               # Configuration files
├── database/
│   ├── migrations/       # Database migrations
│   └── seeders/          # Database seeders
├── public/               # Web server document root
│   └── build/            # Compiled assets
├── resources/
│   ├── css/              # Tailwind CSS
│   ├── js/               # JavaScript
│   ├── views/            # Blade templates
│   └── lang/             # Translations (hr, en)
├── routes/
│   └── web.php           # Web routes
├── storage/
│   ├── app/
│   ├── framework/
│   └── logs/
├── tests/                # PHPUnit tests
├── .env                  # Environment configuration
├── composer.json         # PHP dependencies
├── package.json          # Node dependencies
├── vite.config.js        # Vite configuration
└── README.md
```

### Database Architecture

#### Laravel Tables (Eloquent Models)

- `users` - User authentication
- `contacts` - Contact form submissions
- `sessions` - Session storage
- `cache` - Cache storage
- `jobs` - Queue jobs
- `failed_jobs` - Failed queue jobs

#### Legacy "baza" Tables (DB Facade)

- `baza_realestate` - Core property data with multilingual fields
- `baza_realestate_web` - Web-specific data (URL aliases, active status)
- `baza_locations_all` - Detailed location data
- `baza_locations_main` - Primary location categories
- `baza_realestate_types` - Property type classifications
- `baza_company` - Company/agency information

### Multilingual System

**Supported Languages**: Croatian (hr), English (en)

**URL Structure**:
- Croatian: `/nekretnina/{alias}` (property view)
- English: `/property/{alias}` (property view)
- Language switcher: `GET /locale/{locale}`

**Implementation**:
- Session-based locale storage
- Database columns: `name_hr`, `name_en`, `description_hr`, `description_en`, etc.
- Translation files in `lang/hr/` and `lang/en/`

### Image Serving System

Property images are stored in:
```
baza/files/realestate/{company_id}/{realestate_id}/web_images/
```

**Security Features**:
- Path validation to prevent directory traversal
- Images served through Laravel route with authentication
- Real path verification using `realpath()`

**Helper Functions** (see `app/Helpers.php`):
- `getMainImage($company_id, $realestate_id)` - Get first image URL
- `getAllImages($company_id, $realestate_id)` - Get all image URLs

---

## Development Workflow

### Quick Start Commands

```bash
# Complete setup from scratch
composer setup

# Start all development services
composer dev

# Run tests
composer test

# Code formatting
vendor/bin/pint
```

### Database Operations

```bash
# Run migrations
php artisan migrate

# Fresh database with seeders
php artisan migrate:fresh --seed

# Rollback last migration
php artisan migrate:rollback

# Check migration status
php artisan migrate:status
```

### Asset Build

```bash
# Development with hot reload
npm run dev

# Production build
npm run build
```

### Queue Management

```bash
# Start queue worker
php artisan queue:listen --tries=1

# View failed jobs
php artisan queue:failed

# Retry failed job
php artisan queue:retry {id}

# Clear all failed jobs
php artisan queue:flush
```

### Cache Management

```bash
# Clear all caches
php artisan optimize:clear

# Cache configuration (production)
php artisan config:cache

# Cache routes (production)
php artisan route:cache

# Cache views
php artisan view:cache

# Clear specific cache
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

### Debugging

```bash
# View logs in real-time
php artisan pail

# List all routes
php artisan route:list

# Tinker (REPL)
php artisan tinker
```

---

## Testing

### Running Tests

```bash
# Run all tests
composer test
# or
php artisan test

# Run specific test file
php artisan test tests/Feature/PropertyTest.php

# Run specific test method
php artisan test --filter=test_property_search

# Run with coverage
php artisan test --coverage
```

### Test Database

For integration tests, ensure your test database mirrors the `baza_*` table structure.

Configure in `phpunit.xml`:

```xml
<env name="DB_DATABASE" value="imb_testing"/>
```

---

## Troubleshooting

### Common Issues

#### 1. Permission Errors

```bash
sudo chown -R www-data:www-data storage bootstrap/cache
sudo chmod -R 775 storage bootstrap/cache
```

#### 2. Queue Not Processing

```bash
# Check queue status
php artisan queue:work --once

# Restart supervisor workers
sudo supervisorctl restart imb-worker:*
```

#### 3. Assets Not Loading

```bash
# Clear cache and rebuild
npm run build
php artisan optimize:clear
```

#### 4. Database Connection Error

- Verify `.env` credentials
- Check MySQL service: `sudo systemctl status mysql`
- Test connection: `mysql -u username -p database_name`

#### 5. Images Not Displaying

- Verify `baza/` directory exists and has correct permissions
- Check image paths in database
- Verify web server can access `baza/files/` directory

#### 6. Language Not Switching

- Clear cache: `php artisan cache:clear`
- Check session configuration in `.env`
- Verify `APP_LOCALE` in `.env`

### Logs

```bash
# Laravel logs
tail -f storage/logs/laravel.log

# Nginx access logs
sudo tail -f /var/log/nginx/access.log

# Nginx error logs
sudo tail -f /var/log/nginx/error.log

# PHP-FPM logs
sudo tail -f /var/log/php8.2-fpm.log
```

---

## Performance Optimization

### Production Checklist

- [ ] Set `APP_DEBUG=false` in `.env`
- [ ] Use `CACHE_STORE=redis` and `SESSION_DRIVER=redis`
- [ ] Run `composer install --optimize-autoloader --no-dev`
- [ ] Cache configuration: `php artisan config:cache`
- [ ] Cache routes: `php artisan route:cache`
- [ ] Cache views: `php artisan view:cache`
- [ ] Enable OPcache in PHP configuration
- [ ] Configure CDN for static assets
- [ ] Enable Gzip compression in Nginx
- [ ] Set up database indexing for frequently queried columns

### Recommended Nginx Optimizations

Add to server block:

```nginx
# Gzip compression
gzip on;
gzip_vary on;
gzip_types text/plain text/css application/json application/javascript text/xml application/xml;

# Cache static assets
location ~* \.(jpg|jpeg|png|gif|ico|css|js|svg|woff|woff2)$ {
    expires 1y;
    add_header Cache-Control "public, immutable";
}
```

---

## Contributing

1. Fork the repository
2. Create a feature branch (`git checkout -b feature/amazing-feature`)
3. Commit your changes (`git commit -m 'Add amazing feature'`)
4. Push to the branch (`git push origin feature/amazing-feature`)
5. Open a Pull Request

### Coding Standards

- Follow PSR-12 coding standards
- Use Laravel Pint for code formatting: `vendor/bin/pint`
- Write tests for new features
- Update documentation as needed

---

## Security

### Reporting Vulnerabilities

If you discover a security vulnerability, please email security@imbnekretnine.hr. All security vulnerabilities will be promptly addressed.

### Security Best Practices

- Keep Laravel and all dependencies updated
- Never commit `.env` file
- Use strong database passwords
- Enable HTTPS in production
- Regularly backup database
- Monitor application logs
- Use prepared statements (Eloquent/Query Builder)
- Validate all user input
- Implement rate limiting on forms

---

## License

This project is proprietary software. All rights reserved.

---

## Support

For technical support or questions:
- **Email**: support@imbnekretnine.hr
- **Documentation**: See `CLAUDE.md` for AI assistant guidance
- **Issues**: Report bugs via GitHub Issues

---

## Acknowledgments

- **Laravel Framework** - [https://laravel.com](https://laravel.com)
- **Tailwind CSS** - [https://tailwindcss.com](https://tailwindcss.com)
- **Vite** - [https://vitejs.dev](https://vitejs.dev)

---

**Built with ❤️ for IMB Nekretnine**
