<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Str;

class HomeController extends Controller
{
    //index
    public function index()
    {
        $locale = app()->getLocale() == 'hr' ? 'hr' : 'en';

        // Get featured properties from database
        $featured_properties = DB::table('baza_realestate as r')
            ->join('baza_realestate_web as w', 'r.id', '=', 'w.realestate_id')
            ->leftjoin('baza_locations_all as l', 'r.location', '=', 'l.id')
            ->select(
                'r.id',
                'r.company_id',
                'r.bedrooms',
                'r.bathrooms',
                'r.parking',
                'r.terraces',
                'r.area',
                'r.price',
                'w.order',
                "w.name_{$locale} as title",
                "w.description_{$locale} as description",
                "w.alias_{$locale} as alias",
                "l.hr as location_name"
            )
            ->where('l.location_main_id', 1) //Dubrovnik
            ->where('r.type_code','LIKE','%V%') //Vile
            ->where('w.active', 1)
            ->distinct()
            ->orderBy('w.order', 'ASC')
            ->limit(5)
            ->get();

        // Add main_image path for each property
        foreach ($featured_properties as $property) {
            $property->main_image = getMainImage($property->company_id, $property->id);
            $property->details_url = route('property.'.$locale, ['alias' => $property->alias]);
        }

        return view('index', compact('featured_properties','locale'));
    }

    //change locale/language
    public function changeLocale(Request $request, $locale)
    {
        $current_locale = session('locale','en');
        // Validate locale
        if (in_array($locale, ['en', 'hr'])) {
            session(['locale' => $locale]);
        }
        //search, property url should be changed to the new locale
        //previous url
        $previous_url = url()->previous();
       
     
        if(strpos($previous_url, 'property') !== false || strpos($previous_url, 'nekretnina') !== false){
            $alias = explode('/', $previous_url);
            $alias = $alias[count($alias)-1];

            //findout the translated alias
            $translated_alias = DB::table('baza_realestate_web')
                ->where("alias_{$current_locale}", $alias)
                ->where('active', 1)
                ->first();
                
            if($translated_alias){
                return redirect()->route('property.'.$locale, ['alias' => $translated_alias->{"alias_{$locale}"}]);
            }else{
                return redirect()->route('search.'.$locale);
            }
        }

        if(strpos($previous_url, 'search') !== false || strpos($previous_url, 'pretraga') !== false){
            return redirect()->route('search.'.$locale);
        }
        
        return redirect()->back()->with('error', 'Invalid request');
    }

    public function about()
    {
        return view('about');
    }

    public function terms()
    {
        return view('terms');
    }

    public function privacyPolicy()
    {
        return view('privacy-policy');
    }

    /**
     * Serve real estate image
     *
     * @param int $company_id
     * @param int $realestate_id
     * @param string $filename
     * @return \Illuminate\Http\Response
     */
    public function serveImage($company_id, $realestate_id, $filename)
    {
        $image_path = base_path("baza/files/realestate/{$company_id}/{$realestate_id}/web_images/{$filename}");

        // Security check: ensure the file exists and is within the allowed directory
        if (!file_exists($image_path) || !is_file($image_path)) {
            abort(404);
        }

        // Security check: prevent directory traversal
        $real_path = realpath($image_path);
        $allowed_dir = realpath(base_path("baza/files/realestate/{$company_id}/{$realestate_id}/web_images"));
     
        if ($real_path === false || strpos($real_path, $allowed_dir) !== 0) {
            abort(404);
        }

        // Get file extension and set appropriate content type
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $mime_types = [
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png',
            'gif' => 'image/gif',
            'webp' => 'image/webp',
        ];

        $mime_type = $mime_types[$extension] ?? 'application/octet-stream';

        return Response::file($image_path, [
            'Content-Type' => $mime_type,
        ]);
    }
}
